/*
 * PTEST.C
 *
 * Displays a FLIC file using the EGI player API.
 *
 * Copyright (c) 1996, ITB CompuPhase. You may use/modify/distribute this file
 * or portions of it. It is provided as an example for the use of the EGI
 * player API. There are no waranties on the correct behavior of this program.
 */
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <commdlg.h>

#include <stdlib.h>     /* size_t   */
#include <string.h>     /* memset() */

#include "eplay.h"
#include "ptest.h"

static LPFLIC    lpFlic;
static HWND      hwndFlic;
static HINSTANCE hinst;
static HBITMAP   hbmpTile;
static BOOL      bShowRegion;

static char szAppName[] = "EGI player";

LRESULT CALLBACK _export MainWndProc(HWND hwnd,UINT message,WPARAM wParam,LPARAM lParam);

static void SetClientSize(HWND hwnd, int Width, int Height)
{
  RECT rectClient, rectWindow;

  GetClientRect(hwnd,&rectClient);
  if (rectClient.right!=Width || rectClient.bottom!=Height) {
    int WinWidth, WinHeight;
    GetWindowRect(hwnd,&rectWindow);
    WinWidth  = rectWindow.right  - rectWindow.left;
    WinHeight = rectWindow.bottom - rectWindow.top;
    SetWindowPos(hwnd,0,0,0,
                 WinWidth  + (Width  - rectClient.right),
                 WinHeight + (Height - rectClient.bottom),
                 SWP_NOMOVE|SWP_NOACTIVATE|SWP_NOZORDER);
  } /* if */
}

static BOOL EnableDlgItem(HWND hwnd, int iItem, BOOL bEnable)
{
  HWND hwndDlgItem=GetDlgItem(hwnd,iItem);
  return EnableWindow(hwndDlgItem,bEnable);
}

static int SetPalette(HDC hdc, HPALETTE hpal)
{
  /* Select the new palette in the foreground. */
  HPALETTE hpalOrg = SelectPalette(hdc, hpal, FALSE);
  int result=RealizePalette(hdc);
  /* Select the original palette in the background. Although it is not
   * necessary to realize the original palette afterwards, it does not harm
   * either.
   */
  SelectPalette(hdc, hpalOrg, TRUE);
  RealizePalette(hdc);
  return result;
}

static void SetFlicPalette(HWND hwnd, LPFLIC lpFlic)
{
  HDC hdc = GetDC(hwnd);
  int segment = FlicGetParam(lpFlic, FLIC_PARAM_CURSEGMENT);

  /* Create a palette for the first frame, then select/realize it. */
  FlicNextFrame(lpFlic);
  SetPalette(hdc, FlicGetHandle(lpFlic,0,FLIC_HANDLE_PALETTE));
  /* Reset the FLIC file, clean up */
  FlicSetParam(lpFlic, FLIC_PARAM_CURSEGMENT, segment);
  ReleaseDC(hwnd, hdc);
}

static void TileWindow(HDC hdc,HBITMAP hbmp,int width,int height)
{
  int x,y;
  BITMAP bm;
  HDC hdcMem;

  GetObject(hbmp,sizeof bm,(LPSTR)&bm);
  hdcMem=CreateCompatibleDC(hdc);
  hbmp=SelectObject(hdcMem,hbmp);

  for (y=0; y<height; y+=bm.bmHeight)
    for (x=0; x<width; x+=bm.bmWidth)
      BitBlt(hdc,x,y,bm.bmWidth,bm.bmHeight,hdcMem,0,0,SRCCOPY);

  SelectObject(hdcMem,hbmp);
  DeleteObject(hdcMem);
}

static void DrawMasked(LPFLIC lpFlic,HDC hdcDisplay,HBITMAP hbmpTile)
{
  HDC hdcCanvas,hdcFlic;
  HBITMAP hbmpCanvas,hbmpFlic;
  HPALETTE hpalFlic,hpalCanvas,hpalDisplay;
  int width,height;

  #if defined(WIN32)
    /* This function is called mainly after receiving a FLIC_NOTIFY
     * message, but also upon reception of a WM_PAINT. Therefore it
     * it is still a good idea to use the mutex protection (the
     * WM_PAINT can arrive at any time.
     */
    HANDLE hmutex=FlicGetHandle(lpFlic,NULL,FLIC_HANDLE_MUTEX);
    if (WaitForSingleObject(hmutex,1000)!=WAIT_OBJECT_0)
      return;
  #endif

  /* Create the canvas */
  width=FlicGetParam(lpFlic,FLIC_PARAM_WIDTH);
  height=FlicGetParam(lpFlic,FLIC_PARAM_HEIGHT);
  hdcFlic=CreateCompatibleDC(hdcDisplay);

  hdcCanvas=CreateCompatibleDC(hdcDisplay);
  hpalFlic=FlicGetHandle(lpFlic,0,FLIC_HANDLE_PALETTE);
  hpalCanvas=SelectPalette(hdcCanvas,hpalFlic,FALSE);
  hpalDisplay=SelectPalette(hdcDisplay,hpalFlic,FALSE);
  RealizePalette(hdcDisplay);

  hbmpCanvas=CreateCompatibleBitmap(hdcDisplay,width,height);
  hbmpCanvas=SelectObject(hdcCanvas,hbmpCanvas);

  /* Draw the background */
  TileWindow(hdcCanvas,hbmpTile,width,height);

  /* Mask off the non-transparent areas */
  hbmpFlic=FlicGetHandle(lpFlic,hdcCanvas,FLIC_HANDLE_MASK);
  hbmpFlic=SelectObject(hdcFlic,hbmpFlic);
  BitBlt(hdcCanvas,0,0,width,height,hdcFlic,0,0,SRCAND);
  SelectObject(hdcFlic,hbmpFlic);

  /* Put in FLIC image */
  hbmpFlic=FlicGetHandle(lpFlic,hdcCanvas,FLIC_HANDLE_BITMAP);
  hbmpFlic=SelectObject(hdcFlic,hbmpFlic);
  BitBlt(hdcCanvas,0,0,width,height,hdcFlic,0,0,SRCPAINT);
  SelectObject(hdcFlic,hbmpFlic);

  /* Display the complete new image in the window */
  BitBlt(hdcDisplay,0,0,width,height,hdcCanvas,0,0,SRCCOPY);

  /* Clean up */
  SelectPalette(hdcDisplay,hpalDisplay,TRUE);
  SelectPalette(hdcCanvas,hpalCanvas,TRUE);
  hbmpCanvas=SelectObject(hdcCanvas,hbmpCanvas);
  DeleteObject(hbmpCanvas);
  DeleteObject(hdcCanvas);
  DeleteObject(hdcFlic);

  #if defined(WIN32)
    ReleaseMutex(hmutex);
  #endif
}

static void DrawRegion(LPFLIC lpFlic,HDC hdc)
{
  RECT r;
  BOOL result;

  result=FlicGetRect(lpFlic,FLIC_RECT_FIRST,&r);
  while (result) {
    /* The standard Microsoft Windows function DrawFrameRect() omits
     * a pixel on the right and bottom edges. Therefore, we adjust
     * the rectangle.
     */
    r.right++;
    r.bottom++;
    DrawFocusRect(hdc,&r);
    result=FlicGetRect(lpFlic,FLIC_RECT_NEXT,&r);
  } /* while */
}

static void UpdateMenu(HWND hwnd,LPFLIC lpFlic)
{
  int curmode;
  HMENU hmenu;

  /* Show which display modes are available and what the current mode is */
  if ((hmenu=GetSystemMenu(hwnd,FALSE))==NULL)
    return;
  if (lpFlic==NULL) {
    EnableMenuItem(hmenu,ID_BITBLT,MF_DISABLED|MF_GRAYED);
    EnableMenuItem(hmenu,ID_DIBTODEV,MF_DISABLED|MF_GRAYED);
    EnableMenuItem(hmenu,ID_STRETCHDIB,MF_DISABLED|MF_GRAYED);
    EnableMenuItem(hmenu,ID_WING,MF_DISABLED|MF_GRAYED);
    EnableMenuItem(hmenu,ID_DIBSECTION,MF_DISABLED|MF_GRAYED);
    CheckMenuItem(hmenu,ID_BITBLT,MF_UNCHECKED);
    CheckMenuItem(hmenu,ID_DIBTODEV,MF_UNCHECKED);
    CheckMenuItem(hmenu,ID_STRETCHDIB,MF_UNCHECKED);
    CheckMenuItem(hmenu,ID_WING,MF_UNCHECKED);
    CheckMenuItem(hmenu,ID_DIBSECTION,MF_UNCHECKED);
  } else {
    curmode=FlicGetParam(lpFlic,FLIC_PARAM_MODE);
    EnableMenuItem(hmenu,ID_BITBLT,
                   FlicSetParam(lpFlic,FLIC_PARAM_MODE,FLIC_MODE_BITBLT) ?
                   MF_ENABLED : MF_DISABLED|MF_GRAYED);
    EnableMenuItem(hmenu,ID_DIBTODEV,
                   FlicSetParam(lpFlic,FLIC_PARAM_MODE,FLIC_MODE_DIBTODEV) ?
                   MF_ENABLED : MF_DISABLED|MF_GRAYED);
    EnableMenuItem(hmenu,ID_STRETCHDIB,
                   FlicSetParam(lpFlic,FLIC_PARAM_MODE,FLIC_MODE_STRETCHDIB) ?
                   MF_ENABLED : MF_DISABLED|MF_GRAYED);
    EnableMenuItem(hmenu,ID_WING,
                   FlicSetParam(lpFlic,FLIC_PARAM_MODE,FLIC_MODE_WING) ?
                   MF_ENABLED : MF_DISABLED|MF_GRAYED);
    EnableMenuItem(hmenu,ID_DIBSECTION,
                   FlicSetParam(lpFlic,FLIC_PARAM_MODE,FLIC_MODE_DIBSECTION) ?
                   MF_ENABLED : MF_DISABLED|MF_GRAYED);
    FlicSetParam(lpFlic,FLIC_PARAM_MODE,curmode);
    CheckMenuItem(hmenu,ID_BITBLT,
                  curmode==FLIC_MODE_BITBLT ? MF_CHECKED : MF_UNCHECKED);
    CheckMenuItem(hmenu,ID_DIBTODEV,
                  curmode==FLIC_MODE_DIBTODEV ? MF_CHECKED : MF_UNCHECKED);
    CheckMenuItem(hmenu,ID_STRETCHDIB,
                  curmode==FLIC_MODE_STRETCHDIB ? MF_CHECKED : MF_UNCHECKED);
    CheckMenuItem(hmenu,ID_WING,
                  curmode==FLIC_MODE_WING ? MF_CHECKED : MF_UNCHECKED);
    CheckMenuItem(hmenu,ID_DIBSECTION,
                  curmode==FLIC_MODE_DIBSECTION ? MF_CHECKED : MF_UNCHECKED);
  } /* if */
}

static int Open(HWND hwnd, LPCSTR lpFilename)
{
  if (lstrlen(lpFilename)==0)
    return FALSE;

  if (!hwndFlic) {
    hwndFlic=CreateWindow(szAppName,lpFilename,WS_OVERLAPPED|WS_CAPTION,
                          CW_USEDEFAULT,CW_USEDEFAULT,30,30,hwnd,
                          NULL,hinst,NULL);
    if (hwndFlic!=0) {
      ShowWindow(hwndFlic,SW_SHOW);
      UpdateWindow(hwndFlic);
    } else {
      MessageBox(hwnd,"Error: failed to create animation window",szAppName,MB_OK);
      return FALSE;
    } /* if */
  } /* if */
  if ((lpFlic=FlicOpen(lpFilename))==NULL) {
    MessageBox(NULL, lpFilename, "Cannot open, invalid file", MB_OK);
    return FALSE;
  } /* if */

  if ((FlicGetParam(lpFlic,FLIC_PARAM_MASK) & 1) != 0) {
    /* For transparent FLIC animations, we must do the drawing ourselves. To
     * this end, we switch the automatic painting off, and ask for a
     * notification message for each frame that must be drawn.
     */
    FlicSetParam(lpFlic,FLIC_PARAM_DRAWFRAME,FALSE);
    EnableDlgItem(hwnd,ID_TILE1,TRUE);
    EnableDlgItem(hwnd,ID_TILE2,TRUE);
    EnableDlgItem(hwnd,ID_TILE3,TRUE);
  } /* if */
  /* Both when drawing transparency masks and when showing the region of
   * frame differences, we need a notification for each frame that is
   * decoded and displayed. Hence, just always ask a notification per
   * frame.
   */
  FlicSetParam(lpFlic,FLIC_PARAM_FRNOTIFY,TRUE);
  /* To avoid palette switches between this dialog box window (with the
   * start and stop buttons) and the animation window (hwndFlic), we select
   * the same as the FLIC image into the dialog box. This also has the effect
   * of invalidating the animation window, so it displays the first frame.
   */
  SetFlicPalette(hwnd, lpFlic);
  /* Make the client area of the animation window have the same size as the
   * size of the FLIC file.
   */
  SetClientSize(hwndFlic,FlicGetParam(lpFlic,FLIC_PARAM_WIDTH),
                FlicGetParam(lpFlic,FLIC_PARAM_HEIGHT));
  /* Make sure the tile is shown, in case the animation file contains
   * transparency masks.
   */
  InvalidateRect(hwndFlic,NULL,FALSE);

  EnableDlgItem(hwnd,ID_CLOSE,TRUE);
  EnableDlgItem(hwnd,ID_PLAY,TRUE);
  EnableDlgItem(hwnd,ID_STEP,TRUE);
  EnableDlgItem(hwnd,ID_OPEN,FALSE);

  /* Update the menu */
  UpdateMenu(hwnd,lpFlic);
  return TRUE;
}

#if defined(__BORLANDC__)
#  pragma argsused
#endif
BOOL CALLBACK _export MainDialog(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
  OPENFILENAME ofn;
  char szFilename[256];
  HDC hdc;
  int result;
  char name[20];
  HMENU hsysmenu,hmenu;

  switch (msg) {
  case WM_INITDIALOG:
    EnableDlgItem(hwnd,ID_PLAY,FALSE);
    EnableDlgItem(hwnd,ID_STOP,FALSE);
    EnableDlgItem(hwnd,ID_STEP,FALSE);
    EnableDlgItem(hwnd,ID_CLOSE,FALSE);
    CheckDlgButton(hwnd,ID_TILE1,1);
    EnableDlgItem(hwnd,ID_TILE1,FALSE);
    EnableDlgItem(hwnd,ID_TILE2,FALSE);
    EnableDlgItem(hwnd,ID_TILE3,FALSE);
    /* add a few items to the system menu */
    if ((hsysmenu=GetSystemMenu(hwnd,FALSE)) != NULL) {
      AppendMenu(hsysmenu,MF_SEPARATOR,0,NULL);
      AppendMenu(hsysmenu,MF_STRING,ID_SHOWREGION,"Show 'difference &region'");
      if ((hmenu=LoadMenu(hinst,"PTEST_MENU"))!=NULL)
        AppendMenu(hsysmenu,MF_POPUP,(UINT)hmenu,"Display &mode");
      UpdateMenu(hwnd,NULL);
    } /* if */
    return TRUE;
  case WM_COMMAND:
    switch (LOWORD(wParam)) {           /* LOWORD for 16/32-bit compatibility */
    case IDCANCEL:
      if (lpFlic!=NULL) {
        FlicClose(lpFlic);
        lpFlic=NULL;
        UpdateMenu(hwnd,NULL);
      } /* if */
      EndDialog(hwnd,0);
      return TRUE;
    case ID_OPEN:
      *szFilename='\0';
      memset(&ofn,0,sizeof ofn);
      ofn.lStructSize=sizeof ofn;
      ofn.hwndOwner=hwnd;
      ofn.lpstrFilter="FLIC files\0*.flc;*.fli\0";
      ofn.nFilterIndex=1;
      ofn.lpstrFile=szFilename;
      ofn.nMaxFile=sizeof szFilename;
      ofn.Flags=OFN_FILEMUSTEXIST;
      if (GetOpenFileName(&ofn))
        Open(hwnd,szFilename);
      return TRUE;
    case ID_PLAY:
      if (lpFlic!=NULL && !FlicGetParam(lpFlic,FLIC_PARAM_ISPLAYING)) {
        FlicPlay(lpFlic,hwndFlic);
        EnableDlgItem(hwnd,ID_PLAY,FALSE);
        EnableDlgItem(hwnd,ID_STOP,TRUE);
        EnableDlgItem(hwnd,ID_STEP,FALSE);
        EnableDlgItem(hwnd,ID_OPEN,FALSE);
      } else {
        MessageBeep(0);
      } /* if */
      return TRUE;
    case ID_STOP:
      if (lpFlic!=NULL && FlicStop(lpFlic)) {
        EnableDlgItem(hwnd,ID_PLAY,TRUE);
        EnableDlgItem(hwnd,ID_STOP,FALSE);
        EnableDlgItem(hwnd,ID_STEP,TRUE);
      } else {
        MessageBeep(0);
      } /* if */
      return TRUE;
    case ID_STEP:
      if (lpFlic!=NULL && !FlicGetParam(lpFlic,FLIC_PARAM_ISPLAYING)) {
        hdc=GetDC(hwndFlic);
        result=FlicNextFrame(lpFlic);
        if ((result & FLIC_FRAME_IMG)!=0) {
          if ((FlicGetParam(lpFlic,FLIC_PARAM_MASK) & 1) != 0)
            DrawMasked(lpFlic,hdc,hbmpTile);
          else
            FlicPaint(lpFlic,hdc,FALSE);
          if (bShowRegion)
            DrawRegion(lpFlic,hdc);
        } /* if */
        ReleaseDC(hwndFlic,hdc);
      } else {
        MessageBeep(0);
      } /* if */
      return TRUE;
    case ID_CLOSE:
      if (lpFlic!=NULL) {
        FlicClose(lpFlic);
        lpFlic=NULL;
      } /* if */
      EnableDlgItem(hwnd,ID_OPEN,TRUE);
      EnableDlgItem(hwnd,ID_PLAY,FALSE);
      EnableDlgItem(hwnd,ID_CLOSE,FALSE);
      EnableDlgItem(hwnd,ID_STEP,FALSE);
      EnableDlgItem(hwnd,ID_STOP,FALSE);
      EnableDlgItem(hwnd,ID_TILE1,FALSE);
      EnableDlgItem(hwnd,ID_TILE2,FALSE);
      EnableDlgItem(hwnd,ID_TILE3,FALSE);
      UpdateMenu(hwnd,NULL);
      if (hwndFlic) {
        DestroyWindow(hwndFlic);
        hwndFlic=0;
      } /* if */
      return TRUE;
    case ID_TILE1:
    case ID_TILE2:
    case ID_TILE3:
      if (hbmpTile)
        DeleteObject(hbmpTile);
      wsprintf(name,"TILE%d",LOWORD(wParam)-ID_TILE1+1);
      hbmpTile=LoadBitmap(hinst,name);
      return TRUE;
    } /* switch */
    return FALSE;

  case WM_SYSCOMMAND:
    switch (LOWORD(wParam)) {
    case ID_SHOWREGION:
      bShowRegion= !bShowRegion;
      if ((hsysmenu=GetSystemMenu(hwnd,FALSE)) != NULL)
        CheckMenuItem(hsysmenu,ID_SHOWREGION,
                      bShowRegion ? MF_CHECKED : MF_UNCHECKED);
      return TRUE;
    case ID_BITBLT:
    case ID_DIBTODEV:
    case ID_STRETCHDIB:
    case ID_WING:
    case ID_DIBSECTION:
      if ((hsysmenu=GetSystemMenu(hwnd,FALSE))==NULL || lpFlic==NULL)
        return TRUE;
      result=FlicGetParam(lpFlic,FLIC_PARAM_MODE);
      switch (result) {
      case FLIC_MODE_BITBLT:
        CheckMenuItem(hsysmenu,ID_BITBLT,MF_UNCHECKED);
        break;
      case FLIC_MODE_DIBTODEV:
        CheckMenuItem(hsysmenu,ID_DIBTODEV,MF_UNCHECKED);
        break;
      case FLIC_MODE_STRETCHDIB:
        CheckMenuItem(hsysmenu,ID_STRETCHDIB,MF_UNCHECKED);
        break;
      case FLIC_MODE_WING:
        CheckMenuItem(hsysmenu,ID_WING,MF_UNCHECKED);
        break;
      case FLIC_MODE_DIBSECTION:
        CheckMenuItem(hsysmenu,ID_DIBSECTION,MF_UNCHECKED);
        break;
      } /* switch */
      switch (LOWORD(wParam)) {
      case ID_BITBLT:
        FlicSetParam(lpFlic,FLIC_PARAM_MODE,FLIC_MODE_BITBLT);
        CheckMenuItem(hsysmenu,ID_BITBLT,MF_CHECKED);
        break;
      case ID_DIBTODEV:
        FlicSetParam(lpFlic,FLIC_PARAM_MODE,FLIC_MODE_DIBTODEV);
        CheckMenuItem(hsysmenu,ID_DIBTODEV,MF_CHECKED);
        break;
      case ID_STRETCHDIB:
        FlicSetParam(lpFlic,FLIC_PARAM_MODE,FLIC_MODE_STRETCHDIB);
        CheckMenuItem(hsysmenu,ID_STRETCHDIB,MF_CHECKED);
        break;
      case ID_WING:
        FlicSetParam(lpFlic,FLIC_PARAM_MODE,FLIC_MODE_WING);
        CheckMenuItem(hsysmenu,ID_WING,MF_CHECKED);
        break;
      case ID_DIBSECTION:
        FlicSetParam(lpFlic,FLIC_PARAM_MODE,FLIC_MODE_DIBSECTION);
        CheckMenuItem(hsysmenu,ID_DIBSECTION,MF_CHECKED);
        break;
      } /* switch */
    } /* switch */
    return FALSE;

  } /* switch */

  return FALSE;
}


#if defined(__BORLANDC__)
#  pragma argsused
#endif
int PASCAL WinMain(HINSTANCE hInst,HINSTANCE hPrevInst,LPSTR lpCmdLine,int nCmdShow)
{
  WNDCLASS WndClass;

  if (hPrevInst==0) {
    memset(&WndClass, 0, sizeof WndClass);
    WndClass.style         = 0;
    WndClass.lpfnWndProc   = MainWndProc;
    WndClass.hInstance     = hInst;
    WndClass.hIcon         = LoadIcon(hInst, "appicon");
    WndClass.hCursor       = LoadCursor(NULL, IDC_ARROW);
    WndClass.hbrBackground = GetStockObject(WHITE_BRUSH);
    WndClass.lpszClassName = (LPSTR)szAppName;

    if (!RegisterClass(&WndClass)) {
      MessageBox(NULL,"Error: failed to create window class",szAppName,MB_OK);
      return 0;
    } /* if */
  } /* if */
  hbmpTile=LoadBitmap(hInst,"TILE1");
  hinst=hInst;  /* hinst is global */
  DialogBox(hInst,"PTEST",NULL,MainDialog);
  DeleteObject(hbmpTile);
  return 0;
}

LRESULT CALLBACK _export MainWndProc(HWND hwnd,UINT message,WPARAM wParam,LPARAM lParam)
{
static int loops;
static int segment;
  PAINTSTRUCT ps;
  HDC hdc;

  switch (message) {
  case WM_CREATE:
    /* We play every segment that has a ring frame 10 times. */
    loops=0;
    segment=0;
    break;

  case WM_PALETTECHANGED:
    if ((HWND)wParam==hwnd)
      break;
  case WM_QUERYNEWPALETTE:
    if (lpFlic==NULL || FlicGetHandle(lpFlic,0,FLIC_HANDLE_PALETTE)==0)
      break;
    InvalidateRect(hwnd,NULL,FALSE);
    return TRUE;

  case WM_DESTROY:
    if (lpFlic!=NULL) {
      FlicClose(lpFlic);
      lpFlic=NULL;
    } /* if */
    break;

  case WM_PAINT:
    BeginPaint(hwnd,&ps);
    if (lpFlic!=NULL) {
      if ((FlicGetParam(lpFlic,FLIC_PARAM_MASK) & 1) != 0)
        DrawMasked(lpFlic,ps.hdc,hbmpTile);   /* do transparent drawing yourself */
      else
        FlicPaint(lpFlic,ps.hdc,TRUE);      /* let EPLAY handle the normal case */
      if (bShowRegion)
        DrawRegion(lpFlic,ps.hdc);
    } /* if */
    EndPaint(hwnd,&ps);
    break;

  case FLIC_NOTIFY:
    switch (wParam) {
    case FN_LASTFRAME:
      if (++loops < 10 && FlicGetParam(lpFlic,FLIC_PARAM_CONTINUOUS))
        break;          // play continues segments 10 times, play transitions only once
      segment++;        // go on to the next segment
      loops=0;
      if (segment >= FlicGetParam(lpFlic,FLIC_PARAM_NUMSEGMENTS))
        segment=0;      // after the last segment, wrap back to the first
      FlicSetParam(lpFlic,FLIC_PARAM_CURSEGMENT,segment);
      // if playback stopped because of a segment without ring frame, restart
      if (!FlicGetParam(lpFlic,FLIC_PARAM_ISPLAYING))
        FlicPlay(lpFlic,hwnd);
      break;
    case FN_FRAME:
      hdc=GetDC(hwnd);
      if ((FlicGetParam(lpFlic,FLIC_PARAM_MASK) & 1) != 0)
        DrawMasked(lpFlic,hdc,hbmpTile);
      if (bShowRegion)
        DrawRegion(lpFlic,hdc);
      ReleaseDC(hwnd,hdc);
      break;
    } /* switch */
    break;

  default:
    return DefWindowProc(hwnd,message,wParam,lParam);
  } /* switch */
  return 0L;
}
